'use strict';

/* --------------------------------------------------------------
 statistic_box.js 2016-02-18
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2016 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

/**
 * ## Table Sorting Widget
 *
 * Widget to sort the categories and customers table.
 *
 * ### Example
 *
 * ```html
 * <table data-gx-widget="table_sorting">
 *   <td data-use-table_sorting="true"
 *      data-column="model"
 *      data-section="categories"
 *      data-direction="desc"
 *      data-active-caret="false">
 *    Artikel-Nr.
 *  </td>
 * </table>
 * ```
 *
 * Parameters:
 *   - column: The column which changes the sort order
 *   - section: Section of the table. Example: "categories"
 *   - direction: Ascending or descending. Example: "desc"
 *   - active-caret: Should the caret be added to this element? Example "true"
 *
 * Events:
 *   - Triggering click event on the target element on the mapping hash
 *
 * @module Admin/Widgets/table_sorting
 * @requires jQueryUI-Library
 * @ignore
 */
gx.widgets.module('table_sorting', [], function (data) {

    'use strict';

    // ------------------------------------------------------------------------
    // ELEMENT DEFINITION
    // ------------------------------------------------------------------------

    // Elements

    var $this = $(this),


    // The hidden table row which contains the links for the specific sortings
    hiddenSortbar = 'tr.dataTableHeadingRow_sortbar.hidden',
        caretUp = '<i class="fa fa-caret-up caret"></i>',
        caretDown = '<i class="fa fa-caret-down caret"></i>';

    // ------------------------------------------------------------------------
    // VARIABLE DEFINITION
    // ------------------------------------------------------------------------

    // Widget defaults
    var defaults = {
        elementChildren: '[data-use-table_sorting="true"]',
        caret: '.caret'
    },
        options = $.extend(true, {}, defaults, data),
        module = {};

    // ------------------------------------------------------------------------
    // Mapping hash
    // ------------------------------------------------------------------------

    /**
     * Mappings to the correct links to trigger the table sorting.
     */
    var mapping = {
        categories: {
            sort: {
                asc: 'a.sort',
                desc: 'a.sort-desc'
            },
            name: {
                asc: 'a.name',
                desc: 'a.name-desc'
            },
            model: {
                asc: 'a.model',
                desc: 'a.model-desc'
            },
            stock: {
                asc: 'a.stock',
                desc: 'a.stock-desc'
            },
            status: {
                asc: 'a.status',
                desc: 'a.status-desc'
            },
            startpage: {
                asc: 'a.startpage',
                desc: 'a.startpage-desc'
            },
            price: {
                asc: 'a.price',
                desc: 'a.price-desc'
            },
            discount: {
                asc: 'a.discount',
                desc: 'a.discount-desc'
            }
        },
        customers: {
            lastName: {
                asc: 'a.customers_lastname',
                desc: 'a.customers_lastname-desc'
            },
            firstName: {
                asc: 'a.customers_firstname',
                desc: 'a.customers_firstname-desc'
            },
            dateAccountCreated: {
                asc: 'a.date_account_created',
                desc: 'a.date_account_created-desc'
            },
            dateLastLogon: {
                asc: 'a.date_last_logon',
                desc: 'a.date_last_logon-desc'
            }
        }
    };

    // ------------------------------------------------------------------------
    // PRIVATE METHODS
    // ------------------------------------------------------------------------

    /**
     * Find Target Selector
     *
     * Looks for the target element in the mapping hash and returns the found element.
     *
     * @param {string} section Current section (e.g. 'customers', 'categories', etc.)
     * @param {string} column Column identifier (e.g. 'model', 'price', etc)
     * @param {string} direction Sort direction (e.g. 'asc', 'desc')
     *
     * @throws Error if the element could not be found in the mapping hash
     *
     * @returns {*|jQuery|HTMLElement}
     * @private
     */
    var _findTargetSelector = function _findTargetSelector(section, column, direction) {

        // If the link is available in the mapping hash
        if (section in mapping && column in mapping[section] && direction in mapping[section][column]) {
            // Check the current sort order direction to get the opposite direction
            var targetDirection = direction === 'asc' ? 'desc' : 'asc';

            // The found element from the hash
            var $element = $(hiddenSortbar).find(mapping[section][column][targetDirection]);
            return $element;
        } else {
            throw new Error('Could not find target element');
        }
    };

    /**
     * Open Target Link
     *
     * Maps the column header click events to the correct links.
     *
     * @param event
     * @private
     */
    var _openTargetLink = function _openTargetLink(event) {
        // Clicked element
        var $sourceElement = $(event.target);

        // Retrieve data attributes from element
        var section = $sourceElement.data('section'),
            column = $sourceElement.data('column'),
            direction = $sourceElement.data('direction');

        // Find the correct target selector
        var $targetElement = _findTargetSelector(section, column, direction);

        var targetLink = $targetElement.attr('href');

        // Open the target elements link
        window.open(targetLink, '_self');
    };

    /**
     * Register Children
     *
     * @private
     */
    var _registerChildren = function _registerChildren() {
        $(options.elementChildren).on('click', _openTargetLink);

        // Trigger parent click when caret is clicked
        $(options.caret).on('click', function () {
            $(options.caret).parent().click();
        });
    };

    var _addCaret = function _addCaret() {
        var $activeCaret = $('[data-active-caret="true"]');

        if ($activeCaret.data('direction') === 'asc') {
            $activeCaret.append(caretUp);
        } else {
            $activeCaret.append(caretDown);
        }
    };

    // ------------------------------------------------------------------------
    // INITIALIZATION
    // ------------------------------------------------------------------------

    /**
     * Initialize method of the widget, called by the engine.
     */
    module.init = function (done) {
        _addCaret();
        _registerChildren();
        done();
    };

    // Return data to module engine.
    return module;
});
//# sourceMappingURL=data:application/json;charset=utf8;base64,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
